/* 
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.log4j.appender;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author <a href="mailto:simon_park_mail AT yahoo DOT co DOT uk">Simon Park</a>
 * @version 1.2
 */
final class BackupSuffixHelper {

  BackupSuffixHelper() {
    super();
  }

  final String defaultSuffix() {
    return "1";
  }

  final String next(final String lastLogFilename) {
    final String backupSuffix = this.backupCountFrom(lastLogFilename);
    if ("".equals(backupSuffix)) {
      return this.defaultSuffix();
    }
    int oldBackupSuffixNumber = 0;
    try {
      oldBackupSuffixNumber = Integer.decode(backupSuffix).intValue();
    } catch (NumberFormatException e) {
      return this.defaultSuffix();
    }
    return Integer.toString(oldBackupSuffixNumber + 1);
  }

  private String backupCountFrom(final String logFilename) {
    final LogFileCompressionStrategy[] compressionStrategies = LogFileCompressionStrategy
        .strategies();
    /*
     * Match (1) everything up to the backup count, (2) the count itself, and
     * (3) either the end of input, or a valid compression extension (.zip, .gz,
     * etc) plus the end of input.
     */
    final StringBuffer patternBuffer = new StringBuffer("(.+\\.)([0-9]+)($");
    for (int i = 0; i < compressionStrategies.length; i++) {
      patternBuffer.append("|\\.");
      patternBuffer.append(compressionStrategies[i].getAlgorithmName());
      patternBuffer.append('$');
    }
    patternBuffer.append(')');
    final Pattern pattern = Pattern.compile(patternBuffer.toString());
    final Matcher matcher = pattern.matcher(logFilename);
    if (matcher.find()) {
      return matcher.group(2); // backup count is the second group
    }
    return "";
  }
}
